<?php
App::uses('AppController', 'Controller');
/**
 * Users Controller
 *
 * @property User $User
 * @property PaginatorComponent $Paginator
 * @property SessionComponent $Session
 */
class UsersController extends AppController {

/**
 *	beforeFilter function
 */
	public function beforeFilter() {
		$this->Auth->deny();
		$this->Auth->allow('login', 'register', 'forgot', 'password');
	}

/**
 * login method - User login screen
 */
	public function login() {

		if ($this->request->is('post')) {
			if ($this->Auth->login()) {
				return $this->redirect(array('action' => 'dashboard'));
			} else {
				$this->Session->setFlash(__('Username or password is incorrect'), 'alert-danger');
				$this->request->data['User']['password'] = null;
			}
		}
	}

/**
 * logout method - logout of the system.
 */
	public function logout() {
		$this->Session->setFlash(__('You have successfully logged out.'), 'alert-success');
		$this->redirect($this->Auth->logout());
	}

/**
 * register method
 */
	public function register() {
		if ($this->Auth->user()) {
			$this->Session->setFlash('You are already registered and logged in!', 'alert-warning');
			return $this->redirect('/');
		}

		$this->User->passwordValidator();
		if ($this->request->is('post')) {
			$this->User->create();
			if ($this->User->save($this->request->data)) {
				$this->Session->setFlash(__('Your account has been created. An confirmation email has been sent. Thank you.'), 'alert-success');
				return $this->redirect('/');
			} else {
				$this->request->data['User']['password'] = null;
				$this->request->data['User']['password_confirm'] = null;
				$this->Session->setFlash(__('We could not create your account. Please, try again.'), 'alert-danger');
			}
		}
	}

/**
 * forgot method
 *
 * @return void
 */
	public function forgot() {
		$username = $this->data['User']['email'];
		$this->User->recursive = 0;
		$user = $this->User->find('first', array('conditions' => array('username' => $username)));
		if (empty($user)) {
			$this->Session->setFlash(__('Sorry, "%s" was not found in the system.', $username), 'alert-danger');
		} else {
			$hash = md5($username . time());
			$this->User->id = $user['User']['id'];
			$this->User->saveField('hash_change_password', $hash);
			$this->EmailService->forgot_password($username, $hash);
			$this->Session->setFlash(__('The link to change your password has been sent to your email address.'), 'alert-success');
		}
		$this->redirect('login');
	}

/**
 * password method
 *
 * @throws NotFoundException
 * @param string $hash
 * @return void
 */
	public function password($hash) {
		$this->User->recursive = 0;
		$user = $this->User->findByHashChangePassword($hash);
		if ($user) {
			// remove the hash
			$this->User->id = $user['User']['id'];
			$this->User->saveField('hash_change_password', null);

			$this->Auth->login($user['User']);
			return $this->redirect(array('action' => 'change_password'));
		} else {
			$this->Session->flash(__('Invalid password hash'), 'alert-danger');
		}
	}

/**
 * change_password method
 *
 * @throws NotFoundException
 * @return void
 */
	public function change_password() {
		if ($this->request->is('post') || $this->request->is('put')) {
			$this->User->passwordValidator();
			$this->User->id = $this->Auth->user('id');
			if (!$this->User->exists()) {
				throw new NotFoundException(__('Invalid user'));
			}
			if ($this->User->save($this->request->data)) {
				$this->Session->setFlash(__('Your password has been changed'), 'alert-success');
				return $this->redirect(array('action' => 'dashboard'));
			} else {
				$this->Session->flash(__('Please fix the errors below'), 'alert-danger');
			}
		}
		unset($this->request->data['User']);
	}

/**
 * dashboard method
 */
	public function dashboard() {
		$this->loadModel('Guest');
		$wedding = $this->Guest->Wedding->findById($this->Auth->user('wedding_id'));
		if (empty($wedding)) {
			return $this->redirect(array('action' => 'event'));
		}
		$this->set(compact('wedding'));
	}

/**
 * event method
 */
	public function event() {
		$id = $this->Auth->user('wedding_id');
		$this->loadModel('Wedding');
		if ($this->request->is('post') || $this->request->is('put')) {
			if ($this->Wedding->save($this->request->data)) {
				$this->Session->setFlash(__('The event has been saved.'), 'alert-success');
				$this->Session->write('Auth.User.wedding_id', $this->Wedding->id);
				return $this->redirect(array('action' => 'dashboard'));
			} else {
				$this->Session->setFlash(__('The event could not be saved. Please, try again.'), 'alert-danger');
			}
		} else {
			$options = array('conditions' => array('Wedding.' . $this->Wedding->primaryKey => $id));
			$this->request->data = $this->Wedding->find('first', $options);
		}
	}

/**
 * guests method
 */
	public function guests() {
		$id = $this->Auth->user('id');
		$this->loadModel('Wedding');

		$options = array(
			'conditions' => array(
				'Wedding.user_id' => $id
			),
			'contain' => array('Guest' => array('order' => 'Guest.first_name, Guest.last_name'))
		);
		$this->set('guests', $this->Wedding->find('first', $options));
	}

/**
 * guest_add method
 */
	public function guest_add() {
		$this->loadModel('Guest');
		$wedding = $this->Guest->Wedding->findById($this->Auth->user('wedding_id'));
		if (empty($wedding)) {
			$this->Session->setFlash('Sorry! We could not find that wedding.', 'alert-danger');
			return $this->redirect(array('action' => 'guests'));
		}

		if ($this->request->is('post')) {
			$this->Guest->create();
			if ($this->Guest->save($this->request->data)) {
				$this->Session->setFlash(__('Address  added to the guest list. Thank you.'), 'alert-success');
				return $this->redirect(array('action' => 'guests'));
			} else {
				$this->Session->setFlash(__('We could not save the address. Please, try again.'), 'alert-danger');
			}
		}

		$this->set(compact('wedding'));
	}

/**
 * admin_edit method
 *
 * @throws NotFoundException
 * @param string $id
 * @return void
 */
	public function guest_edit($id = null) {
		$this->loadModel('Guest');
		if (!$this->Guest->exists($id)) {
			throw new NotFoundException(__('Invalid guest'));
		}
		if ($this->request->is('post') || $this->request->is('put')) {
			if ($this->Guest->save($this->request->data)) {
				$this->Session->setFlash(__('The guest has been saved.'), 'alert-success');
				return $this->redirect(array('action' => 'guests'));
			} else {
				$this->Session->setFlash(__('The guest could not be saved. Please, try again.'), 'alert-danger');
			}
		} else {
			$options = array('conditions' => array('Guest.' . $this->Guest->primaryKey => $id));
			$this->request->data = $this->Guest->find('first', $options);
		}
		$wedding = $this->Guest->Wedding->findById($this->Auth->user('wedding_id'));
		$this->set(compact('wedding'));
	}

/**
 * delete method
 *
 * @throws NotFoundException
 * @param string $id
 * @return void
 */
	public function guest_delete($id = null) {
		$this->loadModel('Guest');
		$this->Guest->id = $id;
		if (!$this->Guest->exists()) {
			throw new NotFoundException(__('Invalid guest'));
		}
		$this->request->onlyAllow('post', 'delete');
		if ($this->Guest->delete()) {
			$this->Session->setFlash(__('The guest has been deleted.'), 'alert-warning');
		} else {
			$this->Session->setFlash(__('The guest could not be deleted. Please, try again.'), 'alert-danger');
		}
		return $this->redirect(array('action' => 'guests'));
	}

}
