<?php
App::uses('AppController', 'Controller');
/**
 * Users Controller
 *
 * @property User $User
 * @property PaginatorComponent $Paginator
 * @property SessionComponent $Session
 */
class UsersController extends AppController {

/**
 * test_email
 */
    public function test_email() {
        $this->EmailService->userWelcome($this->Auth->user());
    }

/**
 * beforeFilter function
 */
	public function beforeFilter() {
		$this->Auth->deny();
		$this->Auth->allow('login', 'register', 'forgot', 'password');
	}

/**
 * beforeRender
 */
	public function beforeRender() {
		$this->loadModel('Event');
		$events = $this->Event->getCounts();
		$this->set($events);
	}

/**
 * login method - User login screen
 */
	public function login() {
		if ($this->request->is('post')) {
			if ($this->Auth->login()) {
				return $this->redirect(array('action' => 'dashboard'));
			} else {
				$this->Session->setFlash(__('Username or password is incorrect'), 'alert-danger');
				$this->request->data['User']['password'] = null;
			}
		}
	}

/**
 * logout method - logout of the system.
 */
	public function logout() {
		$this->Session->setFlash(__('You have successfully logged out.'), 'alert-success');
		$this->redirect($this->Auth->logout());
	}

/**
 * register method
 */
	public function register() {
		if ($this->Auth->user()) {
			$this->Session->setFlash('You are already registered and logged in!', 'alert-warning');
			return $this->redirect('/');
		}

		// Load special validation
		$this->User->passwordValidator();
		if ($this->request->is('post')) {
			$this->User->create();
			if ($this->User->save($this->request->data)) {
				$this->Session->setFlash(__('Your account has been created.'), 'alert-success');
				// Log them in
				$this->Auth->login($this->request->data['User']);
                // Send a welcome email
                //$this->EmailService->userWelcome($this->data['User']);
				return $this->redirect(array('action' => 'dashboard'));
			} else {
				$this->request->data['User']['password'] = null;
				$this->request->data['User']['password_confirm'] = null;
				$this->Session->setFlash(__('We could not create your account. Please, try again.'), 'alert-danger');
			}
		}
	}

/**
 * forgot method
 *
 * @return void
 */
	public function forgot() {
		$email = $this->data['User']['email'];
		$this->User->recursive = 0;
		$user = $this->User->find('first', array('conditions' => array('email' => $email)));
		if (empty($user)) {
			$this->Session->setFlash(__('Sorry, "%s" was not found in the system.', $email), 'alert-danger');
		} else {
			$hash = md5($email . time());
			$this->User->id = $user['User']['id'];
			$this->User->saveField('hash_change_password', $hash);
			$this->EmailService->forgotPassword($email, $hash);
			$this->Session->setFlash(__('The link to change your password has been sent to your email address.'), 'alert-success');
		}
		$this->redirect('login');
	}

/**
 * password method
 *
 * @throws NotFoundException
 * @param string $hash
 * @return void
 */
	public function password($hash) {
		$this->User->recursive = 0;
		$user = $this->User->findByHashChangePassword($hash);
		if ($user) {
			// remove the hash
			$this->User->id = $user['User']['id'];
			$this->User->saveField('hash_change_password', null);

			$this->Auth->login($user['User']);
			return $this->redirect(array('action' => 'change_password'));
		} else {
			$this->Session->flash(__('Invalid password hash'), 'alert-danger');
		}
	}

/**
 * change_password method
 *
 * @throws NotFoundException
 * @return void
 */
	public function change_password() {
		if ($this->request->is('post') || $this->request->is('put')) {
			$this->User->passwordValidator();
			$this->User->id = $this->Auth->user('id');
			if (!$this->User->exists()) {
				throw new NotFoundException(__('Invalid user'));
			}
			if ($this->User->save($this->request->data)) {
				$this->Session->setFlash(__('Your password has been changed'), 'alert-success');
				return $this->redirect(array('action' => 'dashboard'));
			} else {
				$this->Session->flash(__('Please fix the errors below'), 'alert-danger');
			}
		}
		unset($this->request->data['User']);
	}

/**
 * dashboard method
 */
	public function dashboard() {
		$this->loadModel('Event');
		$options = array(
			'conditions' => array(
				'user_id' => $this->Auth->user('id'),
				'event_date > NOW()',
			),
			'order' => 'event_date ASC',
			'contain' => 'Guest'
		);
        // admin access
        if ($this->Auth->user('is_admin')) {
            unset($options['conditions']['user_id']);
        }
		$events = $this->Event->find('all', $options);
		$this->set(compact('events'));
	}

/**
 * profile method
 */
    public function profile() {
        if ($this->request->is('post') || $this->request->is('put')) {
            $this->loadModel('User');
            if ($this->User->save($this->request->data)) {
                $this->Session->write('Auth', $this->User->read(null, $this->Auth->User('id')));
                $this->Session->setFlash(__('Your profile has been updated.'), 'alert-success');
            }
        }
        $this->request->data = $this->Auth->user();
    }

/**
 * dashboard method
 */
	public function past_events() {
		$this->loadModel('Event');
		$options = array(
			'conditions' => array(
				'user_id' => $this->Auth->user('id'),
				'event_date <= NOW()',
			),
			'order' => 'event_date ASC',
			'contain' => 'Guest'
		);
        // admin access
        if ($this->Auth->user('is_admin')) {
            unset($options['conditions']['user_id']);
        }
		$events = $this->Event->find('all', $options);
		$this->set(compact('events'));
	}

/**
 * event method
 */
	public function event($id = null, $slug = null) {

		// Handle AJAX requests
		if ($this->request->is('ajax')) {
			$element = $this->request->query['element'];
			$i = $this->request->query['key'];
			$new = $i + 1;
			$this->set(compact('i', 'new'));
			$this->layout = 'ajax';
			$this->render(sprintf('/Elements/%s', $element));
			return;
		}

		$this->loadModel('Event');
		if ($this->request->is('post') || $this->request->is('put')) {
			if ($this->Event->saveAssociated($this->request->data)) {
				if (empty($id)) {
					$this->Session->setFlash(__('Congrats. Copy your Event Link and send it to your guests via email, text or social media.'), 'alert-success');
					$this->Session->write('Auth.User.event_id', $this->Event->id);
				} else {
					$this->Session->setFlash(__('Your event has been updated.'), 'alert-success');
				}
				return $this->redirect(array('action' => 'dashboard'));
			} else {
				$this->Session->setFlash(__('The event could not be saved. Please, try again.'), 'alert-danger');
			}
		} else if (!is_null($id)) {
			$options = array(
				'conditions' => array(
					'Event.id' => $id,
					'Event.slug' => $slug
				),
                'contain' => 'Registry'
			);
			$this->request->data = $this->Event->find('first', $options);
		}
	}

/**
 * guests method
 */
	public function guests($id, $slug) {
		$this->loadModel('Event');
		$options = array(
			'conditions' => array(
				'Event.id' => $id,
				'Event.slug' => $slug
			),
			'contain' => array('Guest' => array('order' => 'Guest.first_name, Guest.last_name'))
		);
		$this->set('guests', $this->Event->find('first', $options));
	}

/**
 * guest_add method
 */
	public function guest_add($id, $slug) {
		$this->loadModel('Guest');
		$options = array(
			'conditions' => array(
				'Event.id' => $id,
				'Event.slug' => $slug
			)
		);
		$event = $this->Guest->Event->find('first', $options);
		if (empty($event)) {
			$this->Session->setFlash('Sorry! We could not find that event.', 'alert-danger');
			return $this->redirect(array('action' => 'dashboard'));
		}

		if ($this->request->is('post')) {
			$this->Guest->create();
			if ($this->Guest->save($this->request->data)) {
				$this->Session->setFlash(__('Address  added to the guest list. Thank you.'), 'alert-success');
				return $this->redirect(array('action' => 'dashboard'));
			} else {
				$this->Session->setFlash(__('We could not save the address. Please, try again.'), 'alert-danger');
			}
		}

		$this->loadModel('State');
		$states = $this->State->find('list', array('fields' => array('State.abbrev', 'State.name', 'Country.name'), 'contain' => 'Country'));

		$this->set(compact('event', 'states'));
	}

/**
 * admin_edit method
 *
 * @throws NotFoundException
 * @param string $id
 * @return void
 */
	public function guest_edit($id = null) {
		$this->loadModel('Guest');
		if (!$this->Guest->exists($id)) {
			throw new NotFoundException(__('Invalid guest'));
		}
		if ($this->request->is('post') || $this->request->is('put')) {
			if ($this->Guest->save($this->request->data)) {
				$this->Session->setFlash(__('The guest has been saved.'), 'alert-success');
				return $this->redirect(array('action' => 'guests'));
			} else {
				$this->Session->setFlash(__('The guest could not be saved. Please, try again.'), 'alert-danger');
			}
		} else {
			$options = array('conditions' => array('Guest.' . $this->Guest->primaryKey => $id));
			$this->request->data = $this->Guest->find('first', $options);
		}
		$event = $this->Guest->Event->findById($this->data['Guest']['event_id']);

		$this->loadModel('State');
		$states = $this->State->find('list', array('fields' => array('State.abbrev', 'State.name', 'Country.name'), 'contain' => 'Country'));

		$this->set(compact('event', 'states'));
	}

/**
 * delete method
 *
 * @throws NotFoundException
 * @param string $id
 * @return void
 */
	public function guest_delete($id = null) {
		$this->loadModel('Guest');
		$this->Guest->id = $id;
		if (!$this->Guest->exists()) {
			throw new NotFoundException(__('Invalid guest'));
		}
		$this->request->onlyAllow('post', 'delete');
		if ($this->Guest->delete()) {
			$this->Session->setFlash(__('The guest has been deleted.'), 'alert-warning');
		} else {
			$this->Session->setFlash(__('The guest could not be deleted. Please, try again.'), 'alert-danger');
		}
		return $this->redirect(array('action' => 'guests'));
	}

}
