<?php
if(!defined('ABSPATH')) {die('You are not allowed to call this page directly.');}

class MeprAuthorizeGateway extends MeprBaseRealGateway {
  public static $order_invoice_str = '_mepr_authnet_order_invoice';
  
  /** Used in the view to identify the gateway */
  public function __construct() {
    $this->name = __("Authorize.net", 'memberpress');
    $this->set_defaults();
    
    $this->capabilities = array(
      'process-credit-cards',
      'process-payments',
      //'process-refunds',
      'create-subscriptions',
      'cancel-subscriptions',
      'update-subscriptions',
      //'suspend-subscriptions',
      //'resume-subscriptions',
      'send-cc-expirations'
    );
    
    // Setup the notification actions for this gateway
    $this->notifiers = array( 'sp' => 'listener' );
    $this->message_pages = array();
  }
  
  public function load($settings) {
    $this->settings = (object)$settings;
    $this->set_defaults();
  }

  public function set_defaults() {
    if(!isset($this->settings))
      $this->settings = array();
    
    $this->settings = (object)array_merge(array('gateway' => get_class($this),
                                                'id' => time() . '-' . uniqid(),
                                                'label' => '',
                                                //'recurrence_type' => '',
                                                'login_name' => '',
                                                'transaction_key' => '',
                                                'logins' => '',
                                                'force_ssl' => false,
                                                'catchup_type' => 'proration',
                                                'debug' => false,
                                                //'use_cron' => false,
                                                'test_mode' => false,
                                                'aimUrl' => '',
                                                'arbUrl' => ''), (array)$this->settings);
    
    $this->id    = $this->settings->id;
    $this->label = $this->settings->label;
    //$this->recurrence_type = $this->settings->recurrence_type;
    $this->hash  = strtoupper(substr(md5($this->id),0,20)); // MD5 hashes used for Silent posts can only be 20 chars long
    
    if($this->is_test_mode()) {
      $this->settings->aimUrl = 'https://test.authorize.net/gateway/transact.dll';
      $this->settings->arbUrl = 'https://apitest.authorize.net/xml/v1/request.api';
    } else {
      $this->settings->aimUrl = 'https://secure.authorize.net/gateway/transact.dll';
      $this->settings->arbUrl = 'https://api.authorize.net/xml/v1/request.api';
    }

    // An attempt to correct people who paste in spaces along with their credentials
    $this->settings->login_name      = trim($this->settings->login_name);
    $this->settings->transaction_key = trim($this->settings->transaction_key);
    $this->settings->logins          = trim($this->settings->logins);
  }

  public function listener() {
    $this->email_status("Silent Post Just Came In (" . $_SERVER['REQUEST_METHOD'] . "):\n" . MeprUtils::object_to_string($_REQUEST, true) . "\n", $this->settings->debug);

    if($this->validate_sp_md5()) {
      if(isset($_REQUEST['x_response_code']) && $_REQUEST['x_response_code'] > 1)
        return $this->record_payment_failure();
      else if(isset($_REQUEST['x_subscription_id']) and !empty($_REQUEST['x_subscription_id'])) {
        $sub = MeprSubscription::get_one_by_subscr_id($_REQUEST['x_subscription_id']);
        if(!$sub) { return false; }
        return $this->record_subscription_payment();
      }
      else if(strtoupper($_REQUEST['x_type']) == 'VOID' || strtoupper($_REQUEST['x_type']) == 'CREDIT')
        return $this->record_refund();

      // Nothing applied so let's bail
      return false;
    }
  }

  public function validate_sp_md5() {
    $logins = array();
    if( !empty($this->settings->logins) ) { $logins = array_map('trim',explode(',',$this->settings->logins)); }
    $logins = array_merge( array('', $this->settings->login_name), $logins );

    //$this->email_status( "Authorize.net names: \n".
    //                     MeprUtils::object_to_string($logins),
    //                     $this->settings->debug );

    // Let's just loop through possible logins (starting with the most likely ... blank)
    // and compare hashes as we go
    foreach($logins as $login) {
      $md5_input = $this->hash.$login.$_REQUEST['x_trans_id'].$_REQUEST['x_amount'];
      $md5 = md5($md5_input);

      //$this->email_status( "Authorize.net, Validate Silent Post: \n".
      //                     "Hash: {$this->hash}\n".
      //                     "Login: {$login} (this should be blank with ARB)\n".
      //                     "x_trans_id: {$_REQUEST['x_trans_id']}\n".
      //                     "x_amount: {$_REQUEST['x_amount']}\n".
      //                     "hash input: {$md5_input}\n".
      //                     "our md5: {$md5}\n".
      //                     "x_MD5_Hash: {$_REQUEST['x_MD5_Hash']}\n".
      //                     "strtoupper comparison: " . strtoupper($md5) . " == " . strtoupper($_REQUEST['x_MD5_Hash']) . "\n",
      //                     $this->settings->debug );

      // Short circuit if we have a match
      if(strtoupper($md5) == strtoupper($_REQUEST['x_MD5_Hash'])) { return true; }
    }

    return false;
  }

  /** Used to send data to a given payment gateway. In gateways which redirect
    * before this step is necessary -- this method should just be left blank.
    */
  public function process_payment($txn) {
    $mepr_options = MeprOptions::fetch();

    if(isset($txn) and $txn instanceof MeprTransaction) {
      $usr = $txn->user();
      $prd = $txn->product();
    }
    else
      throw new MeprGatewayException( __('Payment was unsuccessful, please check your payment details and try again.', 'memberpress') );

    $invoice = $txn->id.'-'.time();

    if( empty($usr->first_name) or empty($usr->last_name) ) {
      $usr->first_name = $_POST['mepr_first_name'];
      $usr->last_name = $_POST['mepr_last_name'];
      $usr->store();
    }

    $args = array( 'x_card_num'       => $_POST['mepr_cc_num'],
                   'x_card_code'      => $_POST['mepr_cvv_code'],
                   'x_exp_date'       => sprintf('%02d',$_POST['mepr_cc_exp_month']).'-'.$_POST['mepr_cc_exp_year'],
                   'x_amount'         => MeprUtils::format_float($txn->amount),
                   'x_description'    => $prd->post_title,
                   'x_invoice_num'    => $invoice,
                   'x_first_name'     => $usr->first_name,
                   'x_last_name'      => $usr->last_name );

    if($mepr_options->show_address_fields) {
      $args = array_merge( array( 'x_address' => get_user_meta($usr->ID, 'mepr-address-one', true),
                                  'x_city'    => get_user_meta($usr->ID, 'mepr-address-city', true),
                                  'x_state'   => get_user_meta($usr->ID, 'mepr-address-state', true),
                                  'x_zip'     => get_user_meta($usr->ID, 'mepr-address-zip', true),
                                  'x_country' => get_user_meta($usr->ID, 'mepr-address-country', true) ), $args );
    }

    $res = $this->send_aim_request('AUTH_CAPTURE', $args);

    $this->email_status("translated AIM response from Authorize.net: \n" . MeprUtils::object_to_string($res, true) . "\n", $this->settings->debug);

    $txn->trans_num = $res['transaction_id'];
    $txn->store();

    $_POST['x_trans_id'] = $res['transaction_id'];
    $_POST['response'] = $res;

    return $this->record_payment();
  }

  /** Used to record a successful recurring payment by the given gateway. It
    * should have the ability to record a successful payment or a failure. It is
    * this method that should be used when receiving an IPN from PayPal or a
    * Silent Post from Authorize.net.
    */
  public function record_subscription_payment() {
    // Make sure there's a valid subscription for this request and this payment hasn't already been recorded
    if( !($sub = MeprSubscription::get_one_by_subscr_id($_POST['x_subscription_id'])) or
        MeprTransaction::get_one_by_trans_num($_POST['x_trans_id']) ) {
      return false;
    }

    $first_txn = $sub->first_txn();

    $txn = new MeprTransaction();
    $txn->amount = MeprUtils::format_float($_POST['x_amount']);
    $txn->user_id = $sub->user_id;
    $txn->product_id = $sub->product_id;
    $txn->txn_type = MeprTransaction::$payment_str;
    $txn->status = MeprTransaction::$complete_str;
    $txn->coupon_id = $first_txn->coupon_id;
    $txn->response = json_encode($_POST);
    $txn->trans_num = $_POST['x_trans_id'];
    $txn->subscription_id = $sub->ID;
    $txn->gateway = $this->id;
    $txn->store();

    $sub->status = MeprSubscription::$active_str;
    $sub->cc_last4 = substr($_POST['x_account_number'],-4); // Don't get the XXXX part of the string
    //$sub->txn_count = $_POST['x_subscription_paynum'];
    $sub->gateway = $this->id;
    $sub->store();

    // Not waiting for a silent post here bro ... just making it happen even
    // though totalOccurrences is Already capped in record_create_subscription()
    $sub->limit_payment_cycles();

    $this->send_transaction_receipt_notices( $txn );
    $this->send_cc_expiration_notices( $txn );

    return $txn;
  }

  /** Used to record a declined payment. */
  public function record_payment_failure() {
    if(isset($_POST['x_trans_id']) and !empty($_POST['x_trans_id'])) {
      $txn_res = MeprTransaction::get_one_by_trans_num($_POST['x_trans_id']);

      if(is_object($txn_res) and isset($txn_res->id)) {
        $txn = new MeprTransaction($txn_res->id);
        $txn->status = MeprTransaction::$failed_str;
        $txn->store();
      }
      else if( isset($_POST['x_subscription_id']) and
               $sub = MeprSubscription::get_one_by_subscr_id($_POST['x_subscription_id']) ) {
        $first_txn = $sub->first_txn();
        $latest_txn = $sub->latest_txn();

        $txn = new MeprTransaction();
        $txn->amount = MeprUtils::format_float($_POST['x_amount']);
        $txn->user_id = $sub->user_id;
        $txn->product_id = $sub->product_id;
        $txn->coupon_id = $first_txn->coupon_id;
        $txn->txn_type = MeprTransaction::$payment_str;
        $txn->status = MeprTransaction::$failed_str;
        $txn->subscription_id = $sub->ID;
        $txn->response = json_encode($_POST);
        $txn->trans_num = $_POST['x_trans_id'];
        $txn->gateway = $this->id;
        $txn->store();

        $sub->status = MeprSubscription::$active_str;
        $sub->gateway = $this->id;
        $sub->expire_txns(); //Expire associated transactions for the old subscription
        $sub->store();
      }
      else
        return false; // Nothing we can do here ... so we outta here

      $this->send_failed_txn_notices($txn);

      return $txn;
    }
    
    return false;
  }
  
  /** Used to record a successful payment by the given gateway. It should have
    * the ability to record a successful payment or a failure. It is this method
    * that should be used when receiving an IPN from PayPal or a Silent Post
    * from Authorize.net.
    */
  public function record_payment()
  {
    if(isset($_POST['x_trans_id']) and !empty($_POST['x_trans_id'])) {
      $obj = MeprTransaction::get_one_by_trans_num($_POST['x_trans_id']);
      
      if(is_object($obj) and isset($obj->id)) {
        $txn = new MeprTransaction();
        $txn->load_data($obj);
        $usr = $txn->user();
        
        // Just short circuit if the transaction has already completed
        if($txn->status == MeprTransaction::$complete_str) { return; }
        
        $txn->status   = MeprTransaction::$complete_str;
        $txn->response = json_encode(
                           array_merge(
                             $_POST['response'],
                             // we need to store this info for auth.net
                             // to be able to handle refunds ... gah
                             array(
                               "cc_last4" => substr($_POST['mepr_cc_num'],-4),
                               "cc_exp_month" => $_POST['mepr_cc_exp_month'],
                               "cc_exp_year" => $_POST['mepr_cc_exp_year']
                             )
                           )
                         );

        // This will only work before maybe_cancel_old_sub is run
        $upgrade = $txn->is_upgrade();
        $downgrade = $txn->is_downgrade();

        $txn->maybe_cancel_old_sub();
        $txn->store();

        $this->email_status("record_payment: Transaction\n" . MeprUtils::object_to_string($txn->rec, true) . "\n", $this->settings->debug);

        $prd = $txn->product();

        if( $prd->period_type=='lifetime' ) {
          if( $upgrade ) {
            $this->upgraded_sub($txn);
            $this->send_upgraded_txn_notices( $txn );
          }
          else if( $downgrade ) {
            $this->downgraded_sub($txn);
            $this->send_downgraded_txn_notices( $txn );
          }
          else {
            $this->new_sub($txn);
          }
         
          $this->send_product_welcome_notices( $txn );
          $this->send_signup_notices( $txn );
        }

        $this->send_transaction_receipt_notices( $txn );
        $this->send_cc_expiration_notices( $txn );

        return $txn;
      }
    }
    
    return false;
  }

  /** This method should be used by the class to record a successful refund from
    * the gateway. This method should also be used by any IPN requests or Silent Posts.
    *
    * Authorize makes this so difficult that I'm disabling this capability from this interface for now
    */
  public function process_refund(MeprTransaction $txn) {
    if( !isset($txn->id) or (int)$txn->id <= 0)
      throw new MeprGatewayException( __('This transaction is invalid.', 'memberpress') );

    if( !empty($txn->response) and $res = json_decode($txn->response) and isset($res->authorization_code) and
        ( ( $sub = $txn->subscription() and 
            !empty($sub->cc_last4) and
            !empty($sub->cc_exp_month) and
            !empty($sub->cc_exp_year) ) or
          ( !empty($res->cc_last4) and
            !empty($res->cc_exp_month) and
            !empty($res->cc_exp_year) ) ) )
    {
      if( !empty($res->cc_last4) and
          !empty($res->cc_exp_month) and
          !empty($res->cc_exp_year) )
      {
        $cc_last4 = $res->cc_last4;
        $cc_exp_month = $res->cc_exp_month;
        $cc_exp_year = $res->cc_exp_year;
      }
      else { // $sub
        $cc_last4 = $sub->cc_last4;
        $cc_exp_month = $sub->cc_exp_month;
        $cc_exp_year = $sub->cc_exp_year;
      }

      $args = array(
        "refId" => $txn->trans_num,
        "transactionRequest" => array(
          "transactionType" => "refundTransaction",
          "amount" => $txn->amount,
          "payment" => array(
            "creditCard" => array(
              'cardNumber' => $cc_last4,
              'expirationDate' => ( sprintf('%02d',$cc_exp_month) . $cc_exp_year )
            )
          ),
          "authCode" => $res->authorization_code
        )
      );

      $this->email_status("refund request: \n" . MeprUtils::object_to_string($args, true) . "\n", $this->settings->debug);

      // ARB hits the same endpoint as AIM XML
      $res = $this->send_arb_request('createTransactionRequest', $args);

      $this->email_status("refund response: \n" . MeprUtils::object_to_string($res, true) . "\n", $this->settings->debug);

      $_REQUEST['x_type'] = 'CREDIT';
      $_POST['x_trans_id'] = $txn->id;
      $_POST['x_invoice_num'] = "{$txn->id}-0000";
      $_POST['x_amount'] = $txn->amount;

      return $this->record_refund();
    }
    else
      throw new MeprGatewayException( __('This transaction can\'t be refunded in MemberPress, please refund through your Virtual Terminal.', 'memberpress') );
  }
  
  /** This method should be used by the class to record a successful refund from
    * the gateway. This method should also be used by any IPN requests or Silent Posts.
    */
  public function record_refund() {
    if(strtoupper($_REQUEST['x_type']) == 'CREDIT') {
      // This is all we've got to reference the old sale in a credit
      if(!isset($_POST['x_invoice_num'])) { return false; }

      preg_match('#^(\d+)-#',$_POST['x_invoice_num'],$m);
      $txn_id = $m[1];
      $txn_res = MeprTransaction::get_one($txn_id);
    }
    else if(strtoupper($_REQUEST['x_type']) == 'VOID')
      $txn_res = MeprTransaction::get_one_by_trans_num($_POST['x_trans_id']);

    if(!isset($txn_res) or empty($txn_res)) { return false; }

    $txn = new MeprTransaction($txn_res->id);

    // Seriously ... if txn was already refunded what are we doing here?
    if($txn->status == MeprTransaction::$refunded_str) { return $txn->id; }

    $returned_amount = MeprUtils::format_float($_POST['x_amount']);
    $current_amount = MeprUtils::format_float($txn->amount);

    if(strtoupper($_POST['x_type']) == 'CREDIT' and $returned_amount < $current_amount ) {
      $txn->amount = $current_amount - $returned_amount;
      $txn->status = MeprTransaction::$complete_str;
    }
    else
      $txn->status = MeprTransaction::$refunded_str;

    $txn->store();

    $this->send_refunded_txn_notices($txn);

    return $txn->id;
  }
  
  /** Used to send subscription data to a given payment gateway. In gateways
    * which redirect before this step is necessary this method should just be
    * left blank.
    */
  public function process_create_subscription($txn) {
    $mepr_options = MeprOptions::fetch();

    if(isset($txn) and $txn instanceof MeprTransaction) {
      $usr = $txn->user();
      $prd = $txn->product();
      $sub = $txn->subscription();
    }
    else
      throw new MeprGatewayException( __('Payment was unsuccessful, please check your payment details and try again.', 'memberpress') );

    //$invoice = $txn->id.'-'.time();
    $invoice = $this->create_new_order_invoice($sub);

    if( empty($usr->first_name) or empty($usr->last_name) ) {
      $usr->first_name = $_POST['mepr_first_name'];
      $usr->last_name = $_POST['mepr_last_name'];
      $usr->store();
    }

    // Default to 9999 for infinite occurrences
    $total_occurrences = $sub->limit_cycles ? $sub->limit_cycles_num : 9999;

    $args = array( "refId" => $invoice,
                   "subscription" => array(
                     "name" => $prd->post_title,
                     "paymentSchedule" => array(
                       "interval" => $this->arb_subscription_interval($sub),
                       // Since Authorize doesn't allow trials that have a different period_type
                       // from the subscription itself we have to do our trials here manually
                       "startDate" => MeprUtils::get_date_from_ts((time()+($sub->trial?MeprUtils::days($sub->trial_days):0)), 'Y-m-d'),
                       "totalOccurrences" => $total_occurrences,
                     ),
                     "amount" => MeprUtils::format_float($txn->amount),
                     "payment" => array(
                       "creditCard" => array(
                         "cardNumber" => $_POST['mepr_cc_num'],
                         "expirationDate" => $_POST['mepr_cc_exp_month'].'-'.$_POST['mepr_cc_exp_year'],
                         "cardCode" => $_POST['mepr_cvv_code']
                       )
                     ),
                     "order" => array(
                       "invoiceNumber" => $invoice,
                       "description" => $prd->post_title
                     ),
                     "billTo" => array(
                       "firstName" => $usr->first_name,
                       "lastName" => $usr->last_name
                     )
                   )
                 );

    if($mepr_options->show_address_fields) {
      $args['subscription']['billTo'] =
        array_merge($args['subscription']['billTo'],
                    array("address" => get_user_meta($usr->ID, 'mepr-address-one', true),
                          "city" => get_user_meta($usr->ID, 'mepr-address-city', true),
                          "state" => get_user_meta($usr->ID, 'mepr-address-state', true),
                          "zip" => get_user_meta($usr->ID, 'mepr-address-zip', true),
                          "country" => get_user_meta($usr->ID, 'mepr-address-country', true)));
    }

    $res = $this->send_arb_request('ARBCreateSubscriptionRequest', $args);

    $_POST['txn_id'] = $txn->id;
    $_POST['subscr_id'] = $res->subscriptionId;

    return $this->record_create_subscription();
  }
  
  /** Used to record a successful subscription by the given gateway. It should have
    * the ability to record a successful subscription or a failure. It is this method
    * that should be used when receiving an IPN from PayPal or a Silent Post
    * from Authorize.net.
    */
  public function record_create_subscription() {
    if(isset($_POST['txn_id']) and is_numeric($_POST['txn_id'])) {
      $txn = new MeprTransaction($_POST['txn_id']);
      $sub = $txn->subscription();
      $sub->subscr_id = $_POST['subscr_id'];
      $sub->status=MeprSubscription::$active_str;
      $sub->created_at = date('c');
      $sub->response = MeprUtils::object_to_string($sub);
      $sub->cc_last4 = substr($_POST['mepr_cc_num'],-4); // Seriously ... only grab the last 4 digits!
      $sub->cc_exp_month = $_POST['mepr_cc_exp_month'];
      $sub->cc_exp_year = $_POST['mepr_cc_exp_year'];
      $sub->store();

      // This will only work before maybe_cancel_old_sub is run
      $upgrade   = $sub->is_upgrade();
      $downgrade = $sub->is_downgrade();

      $sub->maybe_cancel_old_sub();

      $old_amount = $txn->amount; // Save for later

      // We do manual trials in Authorize.net because their trial stuff
      // doesn't allow for trial period_types that differ from the
      // subscription period_types ... and we only support day based trials
      if($sub->trial and $sub->trial_amount > 0.00) {
        // On the bright side, we can bill trials using AIM instantly so
        // there's no need for a subscription_confirmation transaction
        $txn->amount = $sub->trial_amount;
        $txn->txn_type = MeprTransaction::$payment_str;
        $txn->status=MeprTransaction::$pending_str;
        $txn->expires_at=MeprUtils::ts_to_mysql_date(time()+MeprUtils::days($sub->trial_days));
        $txn->store();
        $this->process_payment($txn);
      }
      else { // Setup the confirmation txn to make sure users can log in until their first payment processes
        $mepr_options = MeprOptions::fetch();

        // If trial amount is zero then we've got to make
        // sure the confirmation txn lasts through the trial
        $day_count = ( $sub->trial ? $sub->trial_days : $mepr_options->grace_init_days );
        $txn->expires_at=MeprUtils::ts_to_mysql_date(time()+MeprUtils::days($day_count)); // Grace period before txn processes
        $txn->txn_type=MeprTransaction::$subscription_confirmation_str;
        $txn->status=MeprTransaction::$confirmed_str;
        $txn->trans_num = $sub->subscr_id;
        $txn->amount=0.00; // This txn is just a confirmation txn ... it shouldn't have a cost
        $txn->store();
      }

      // Artificially set the txn amount for the notifications
      $txn->amount=$old_amount;

      if($upgrade) {
        $this->upgraded_sub($sub);
        $this->send_upgraded_sub_notices($sub);
      }
      else if($downgrade) {
        $this->downgraded_sub($sub);
        $this->send_downgraded_sub_notices($sub);
      }
      else {
        $this->new_sub($sub);
        $this->send_new_sub_notices($sub);
      }

      /// This will only send if there's a new signup
      $this->send_product_welcome_notices($txn);
      $this->send_signup_notices( $txn );
    }
  }
  
  /** Used to cancel a subscription by the given gateway. This method should be used
    * by the class to record a successful cancellation from the gateway. This method
    * should also be used by any IPN requests or Silent Posts.
    */
  public function process_update_subscription($sub_id) {
    $mepr_options = MeprOptions::fetch();

    $sub = new MeprSubscription($sub_id);
    if(!isset($sub->ID) || (int)$sub->ID <= 0)
      throw new MeprGatewayException( __('Your payment details are invalid, please check them and try again.', 'memberpress') );

    $usr = $sub->user();
    if(!isset($usr->ID) || (int)$usr->ID <= 0)
      throw new MeprGatewayException( __('Your payment details are invalid, please check them and try again.', 'memberpress') );

    $args = array( "refId" => $sub->ID,
                   "subscriptionId" => $sub->subscr_id,
                   "subscription" => array(
                     "payment" => array(
                       "creditCard" => array(
                         "cardNumber" => $_POST['update_cc_num'],
                         "expirationDate" => $_POST['update_cc_exp_month'].'-'.$_POST['update_cc_exp_year'],
                         "cardCode" => $_POST['update_cvv_code']
                       )
                     ),
                     "billTo" => array(
                       "firstName" => $usr->first_name,
                       "lastName" => $usr->last_name
                     )
                   )
                 );

    if($mepr_options->show_address_fields) {
      $args['subscription']['billTo'] =
        array_merge($args['subscription']['billTo'],
                    array("address" => get_user_meta($usr->ID, 'mepr-address-one', true),
                          "city" => get_user_meta($usr->ID, 'mepr-address-city', true),
                          "state" => get_user_meta($usr->ID, 'mepr-address-state', true),
                          "zip" => get_user_meta($usr->ID, 'mepr-address-zip', true),
                          "country" => get_user_meta($usr->ID, 'mepr-address-country', true)));
    }

    $res = $this->send_arb_request('ARBUpdateSubscriptionRequest', $args);
    
    // Calculate ARB Catch up payment
    if( $sub->is_expired() and $catchup = $sub->calculate_catchup($this->settings->catchup_type) and $catchup->proration > 0.00 ) {
      // Create Transaction
      $txn = new MeprTransaction();
      $txn->subscription_id = $sub->ID;
      $txn->user_id = $sub->user_id;
      $txn->amount = $catchup->proration;
      $txn->prorated = true;
      $txn->product_id = $sub->product_id;
      $txn->gateway = $this->id;

      $now = time();
      $txn->created_at = MeprUtils::ts_to_mysql_date($now);
      $txn->expires_at = MeprUtils::ts_to_mysql_date($catchup->next_billing);

      $txn->store();

      // Bill Catch-Up Payment
      $_POST['mepr_cc_num'] = $_POST['update_cc_num'];
      $_POST['mepr_cvv_code'] = $_POST['update_cvv_code'];
      $_POST['mepr_cc_exp_month'] = $_POST['update_cc_exp_month'];
      $_POST['mepr_cc_exp_year'] = $_POST['update_cc_exp_year'];

      $this->process_payment($txn);
    }

    return $res;
  }

  /** This method should be used by the class to record a successful cancellation
    * from the gateway. This method should also be used by any IPN requests or 
    * Silent Posts.
    */
  public function record_update_subscription() {
    // I don't think we need to do anything here
  }

  /** Used to suspend a subscription by the given gateway.
    */
  public function process_suspend_subscription($sub_id) {}
  
  /** This method should be used by the class to record a successful suspension
    * from the gateway.
    */
  public function record_suspend_subscription() {}

  /** Used to suspend a subscription by the given gateway.
    */
  public function process_resume_subscription($sub_id) {}
  
  /** This method should be used by the class to record a successful resuming of
    * as subscription from the gateway.
    */
  public function record_resume_subscription() {}

  /** Used to cancel a subscription by the given gateway. This method should be used
    * by the class to record a successful cancellation from the gateway. This method
    * should also be used by any IPN requests or Silent Posts.
    */
  public function process_cancel_subscription($sub_id) {
    $sub = new MeprSubscription($sub_id);

    if(!isset($sub->ID) || (int)$sub->ID <= 0)
      throw new MeprGatewayException( __('This subscription is invalid.', 'memberpress') );

    // Should already expire naturally at authorize.net so we have no need
    // to do this when we're "cancelling" because of a natural expiration
    if(!isset($_REQUEST['expire'])) {
      $args = array( "refId" => $sub->ID, "subscriptionId" => $sub->subscr_id );
      $res = $this->send_arb_request('ARBCancelSubscriptionRequest', $args);
    }

    $_POST['subscr_ID'] = $sub->ID;
    return $this->record_cancel_subscription();
  }

  /** This method should be used by the class to record a successful cancellation
    * from the gateway. This method should also be used by any IPN requests or 
    * Silent Posts.
    */
  public function record_cancel_subscription() {
    $subscr_ID = (isset($_POST['subscr_ID']))?$_POST['subscr_ID']:null;
    $sub = new MeprSubscription($subscr_ID);

    if(!isset($sub->ID) || $sub->ID <= 0) { return false; }

    // Seriously ... if sub was already cancelled what are we doing here?
    if($sub->status == MeprSubscription::$cancelled_str) { return true; }

    $sub->status = MeprSubscription::$cancelled_str;
    $sub->store();

    if(isset($_REQUEST['expire']))
      $sub->limit_reached_actions();

    if(!isset($_REQUEST['silent']) || ($_REQUEST['silent']==false))
      $this->send_cancelled_sub_notices($sub);

    return true;
  }

  /** This gets called on the 'init' hook when the signup form is processed ...
    * this is in place so that payment solutions like paypal can redirect
    * before any content is rendered.
    */
  public function process_signup_form($txn) {
    if($txn->amount <= 0.00) {
      MeprTransaction::create_free_transaction($txn);
      return;
    }
  }
  
  /** This gets called on wp_enqueue_script and enqueues a set of
    * scripts for use on the page containing the payment form
    */
  public function enqueue_payment_form_scripts() {
    // No need for this with Authorize.net
  }
  
  /** This spits out html for the payment form on the registration / payment
    * page for the user to fill out for payment. If we're using an offsite
    * payment solution like PayPal then this method will just redirect to it.
    */
  public function display_payment_form($amount, $usr, $product_id, $txn_id) {
    $prd = new MeprProduct($product_id);
    $coupon = false;
    $mepr_options = MeprOptions::fetch();

    $txn = new MeprTransaction($txn_id);
    $usr = $txn->user();
    
    //Artifically set the price of the $prd in case a coupon was used
    if($prd->price != $amount)
    {
      $coupon = true;
      $prd->price = $amount;
    }
    ?>
    <div class="mepr_signup_table">
      <form action="" method="post" id="mepr_authorize_net_payment_form">
        <input type="hidden" name="mepr_process_payment_form" value="Y" />
        <input type="hidden" name="mepr_transaction_id" value="<?php echo $txn_id; ?>" />

        <strong><?php _e('Please enter your Credit Card information below', 'memberpress'); ?></strong><br/><br/>

        <div class="mepr_signup_table_row">
          <label><?php _e('Price', 'memberpress'); ?></label>
          <?php echo MeprTransactionsHelper::format_currency($txn); ?>
        </div>

        <?php // Authorize requires a firstname / lastname so if it's hidden on the signup form ...
              // guess what, the user will still have to fill it out here ?>
        <?php if( empty($usr->first_name) or empty($usr->last_name) ): ?>
          <div class="mepr_signup_table_row">
            <label><?php _e('First Name', 'memberpress'); ?></label>
            <input type="text" name="mepr_first_name" class="mepr-form-input" value="<?php echo (isset($_POST['mepr_first_name']))?$_POST['mepr_first_name']:$usr->first_name; ?>" />
          </div>

          <div class="mepr_signup_table_row">
            <label><?php _e('Last Name', 'memberpress'); ?></label>
            <input type="text" name="mepr_last_name" class="mepr-form-input" value="<?php echo (isset($_POST['mepr_last_name']))?$_POST['mepr_last_name']:$usr->last_name; ?>" />
          </div>
        <?php endif; ?> 

        <div class="mepr_signup_table_row">
          <label><?php _e('Credit Card #', 'memberpress'); ?></label>
          <input type="text" name="mepr_cc_num" class="mepr-form-input" value="<?php echo (isset($_POST['mepr_cc_num']))?$_POST['mepr_cc_num']:''; ?>" autocomplete="off" />
        </div>

        <div class="mepr_signup_table_row">
          <label><?php _e('Expires', 'memberpress'); ?></label>
          <?php $this->months_dropdown('mepr_cc_exp_month','',isset($_POST['mepr_cc_exp_month'])?$_POST['mepr_cc_exp_month']:''); ?>
          <?php $this->years_dropdown('mepr_cc_exp_year','',isset($_POST['mepr_cc_exp_year'])?$_POST['mepr_cc_exp_year']:''); ?>
        </div>

        <div class="mepr_signup_table_row">
          <label><?php _e('CVV Code', 'memberpress'); ?></label>
          <input type="text" name="mepr_cvv_code" class="mepr-form-input" maxlength="5" value="<?php echo (isset($_POST['mepr_cvv_code']))?$_POST['mepr_cvv_code']:''; ?>" />
        </div>

        <input type="submit" class="submit-button mepr_front_button" value="<?php _e('Submit', 'memberpress'); ?>" />&nbsp;<img src="<?php echo admin_url('images/loading.gif'); ?>" style="display: none;" class="mepr-loading-gif" />
      </form>
    </div>
    <?php
  }
  
  /** Validates the payment form before a payment is processed */
  public function validate_payment_form($errors) {
    $mepr_options = MeprOptions::fetch();

    if(!isset($_POST['mepr_transaction_id']) || !is_numeric($_POST['mepr_transaction_id']))
      $errors[] = __('An unknown error has occurred.', 'memberpress');

    // Authorize requires a firstname / lastname so if it's hidden on the signup form ...
    // guess what, the user will still have to fill it out here
    if( !$mepr_options->show_fname_lname and
        ( !isset($_POST['mepr_first_name']) or empty($_POST['mepr_first_name']) or
          !isset($_POST['mepr_last_name']) or empty($_POST['mepr_last_name']) ) )
      $errors[] = __('Your first name and last name must not be blank.', 'memberpress');

    if(!isset($_POST['mepr_cc_num']) || empty($_POST['mepr_cc_num']))
      $errors[] = __('You must enter your Credit Card number.', 'memberpress');
    elseif(!$this->is_credit_card_valid($_POST['mepr_cc_num']))
      $errors[] = __('Your credit card number is invalid.', 'memberpress');

    if(!isset($_POST['mepr_cvv_code']) || empty($_POST['mepr_cvv_code']))
      $errors[] = __('You must enter your CVV code.', 'memberpress');

    return $errors;
  }
  
  /** Displays the form for the given payment gateway on the MemberPress Options page */
  public function display_options_form() {
    $mepr_options = MeprOptions::fetch();
    
    $login_name   = trim($this->settings->login_name);
    $txn_key      = trim($this->settings->transaction_key);
    $logins       = trim($this->settings->logins);
    $test_mode    = ($this->settings->test_mode == 'on' or $this->settings->test_mode == true);
    $debug        = ($this->settings->debug == 'on' or $this->settings->debug == true);
    $force_ssl    = ($this->settings->force_ssl == 'on' or $this->settings->force_ssl == true);
    $catchup_type = $this->settings->catchup_type;
    // $use_cron     = ($this->settings->use_cron == 'on' or $this->settings->use_cron == true);
    
    ?>
    <div class="mepr-options-pane">
      <table>
        <tr>
          <td><?php _e('Login Name:', 'memberpress'); ?></td>
          <td><input type="text" class="regular-text mepr-auto-trim" name="<?php echo $mepr_options->integrations_str; ?>[<?php echo $this->id;?>][login_name]" width="100%" value="<?php echo $login_name; ?>" /></td>
        </tr>
        <tr>
          <td><?php _e('Transaction Key:', 'memberpress'); ?></td>
          <td><input type="text" class="regular-text mepr-auto-trim" name="<?php echo $mepr_options->integrations_str; ?>[<?php echo $this->id;?>][transaction_key]" width="100%" value="<?php echo $txn_key; ?>" /></td>
        </tr>
        <tr>
          <td><?php _e('Virtual Terminal Logins (optional):', 'memberpress'); ?></td>
          <td><input type="text" class="regular-text mepr-auto-trim" name="<?php echo $mepr_options->integrations_str; ?>[<?php echo $this->id;?>][logins]" width="100%" value="<?php echo $logins; ?>" /></td>
        </tr>
        <tr>
          <td colspan="2"><input type="checkbox" name="<?php echo $mepr_options->integrations_str; ?>[<?php echo $this->id;?>][test_mode]"<?php checked($test_mode); ?> />&nbsp;<?php _e('Use Authorize.net Sandbox', 'memberpress'); ?></td>
        </tr>
        <tr>
          <td colspan="2"><input type="checkbox" name="<?php echo $mepr_options->integrations_str; ?>[<?php echo $this->id;?>][debug]"<?php checked($debug); ?> />&nbsp;<?php _e('Send Authorize.net Debug Emails', 'memberpress'); ?></td>
        </tr>
        <tr>
          <td colspan="2"><input type="checkbox" name="<?php echo $mepr_options->integrations_str; ?>[<?php echo $this->id;?>][force_ssl]"<?php checked($force_ssl); ?> />&nbsp;<?php _e('Force SSL', 'memberpress'); ?></td>
        </tr>
        <tr>
          <td><?php _e('Catch Up Payment Type', 'memberpress'); ?></td>
          <td>
            <select name="<?php echo $mepr_options->integrations_str; ?>[<?php echo $this->id;?>][catchup_type]">
              <option value="none" <?php selected($catchup_type, 'none'); ?>><?php _e('None', 'memberpress'); ?></option>
              <option value="period" <?php selected($catchup_type, 'period'); ?>><?php _e('Period', 'memberpress'); ?></option>
              <option value="proration" <?php selected($catchup_type, 'proration'); ?>><?php _e('Proration', 'memberpress'); ?></option>
              <option value="full" <?php selected($catchup_type, 'full'); ?>><?php _e('Full', 'memberpress'); ?></option>
            </select>
          </td>
        </tr>
        <tr>
          <td><?php _e('Silent Post URL:', 'memberpress'); ?></td>
          <td><input type="text" onfocus="this.select();" onclick="this.select();" readonly="true" class="clippy_input regular-text" value="<?php echo $this->notify_url('sp'); ?>" /><span class="clippy"><?php echo $this->notify_url('sp'); ?></span></td>
        </tr>
        <tr>
          <td><?php _e('MD5 Hash Value:', 'memberpress'); ?></td>
          <td><input type="text" onfocus="this.select();" onclick="this.select();" readonly="true" class="clippy_input regular-text" value="<?php echo $this->hash; ?>" /><span class="clippy"><?php echo $this->hash; ?></span></td>
        </tr>
      </table>

    </div>
    <?php
  }
  
  /** Validates the form for the given payment gateway on the MemberPress Options page */
  public function validate_options_form($errors) {
    $mepr_options = MeprOptions::fetch();
    
    if( !isset($_POST[$mepr_options->integrations_str][$this->id]['login_name']) or
        empty($_POST[$mepr_options->integrations_str][$this->id]['login_name']) )
      $errors[] = __("Login Name field cannot be blank.", 'memberpress');
    
    if( !isset($_POST[$mepr_options->integrations_str][$this->id]['transaction_key']) or
        empty($_POST[$mepr_options->integrations_str][$this->id]['transaction_key']) )
      $errors[] = __("Transaction Key field cannot be blank.", 'memberpress');
    
    return $errors;
  }

  /** Displays the update account form on the subscription account page **/
  public function display_update_account_form($sub_id, $errors=array(), $message='') {
    $sub = new MeprSubscription($sub_id);

    require(MEPR_VIEWS_PATH . '/shared/errors.php');

    $last4 = isset($_POST['update_cc_num']) ? substr($_POST['update_cc_num'],-4) : $sub->cc_last4;
    $exp_month = isset($_POST['update_cc_exp_month']) ? $_POST['update_cc_exp_month'] : $sub->cc_exp_month;
    $exp_year = isset($_POST['update_cc_exp_year']) ? $_POST['update_cc_exp_year'] : $sub->cc_exp_year;

    // Only include the full cc number if there are errors
    if(strtolower($_SERVER['REQUEST_METHOD'])=='post' and empty($errors)) {
      $sub->cc_last4 = $last4;
      $sub->cc_exp_month = $exp_month;
      $sub->cc_exp_year = $exp_year;
      $sub->store();

      unset($_POST['update_cvv_code']); // Unset this for security
    }
    else // If there are errors then show the full cc num ... if it's there
      $last4 = isset($_POST['update_cc_num']) ? $_POST['update_cc_num'] : $sub->cc_last4;

    $ccv_code = (isset($_POST['update_cvv_code']))?$_POST['update_cvv_code']:'';

    ?>
    <div class="mepr_update_account_table">
      <?php if( $sub->is_expired() and $catchup = $sub->calculate_catchup($this->settings->catchup_type) and $catchup->proration > 0.00 ): ?>
        <div class="error"><?php printf(__('Note: Because your subscription is expired, when you update your credit card number our system will attempt to bill your card for the prorated amount of %s to catch you up until the next automatic billing.','memberpress'), MeprAppHelper::format_currency($catchup->proration)); ?></div>
      <?php endif; ?>
      <form action="" method="post" id="mepr_authorize_net_update_cc_form">
        <input type="hidden" name="_mepr_nonce" value="<?php echo wp_create_nonce('mepr_process_update_account_form'); ?>" />

        <strong><?php _e('Update your Credit Card information below', 'memberpress'); ?></strong><br/><br/>

        <div class="mepr_update_account_table_row">
          <label><?php _e('Credit Card #', 'memberpress'); ?></label>
          <input type="text" name="update_cc_num" class="mepr-form-input" value="<?php echo MeprUtils::cc_num($last4); ?>" autocomplete="off" />
        </div>

        <div class="mepr_update_account_table_row">
          <label><?php _e( 'Expires', 'memberpress' ); ?></label>
          <?php $this->months_dropdown( 'update_cc_exp_month', '', $exp_month ); ?>
          <?php $this->years_dropdown( 'update_cc_exp_year', '', $exp_year ); ?>
        </div>

        <div class="mepr_update_account_table_row">
          <label><?php _e('CVV Code', 'memberpress'); ?></label>
          <input type="text" name="update_cvv_code" class="mepr-form-input" maxlength="5" value="<?php echo $ccv_code; ?>" />
        </div>

        <input type="submit" class="submit-button mepr_front_button" value="<?php _e('Update Credit Card', 'memberpress'); ?>" />&nbsp;<img src="<?php echo admin_url('images/loading.gif'); ?>" style="display: none;" class="mepr-loading-gif" />
      </form>
    </div>
    <?php
  }

  /** Validates the payment form before a payment is processed */
  public function validate_update_account_form($errors=array()) {
    if( !isset($_POST['_mepr_nonce']) or empty($_POST['_mepr_nonce']) or
        !wp_verify_nonce($_POST['_mepr_nonce'], 'mepr_process_update_account_form') )
      $errors[] = __('An unknown error has occurred. Please try again.', 'memberpress');
    
    if(!isset($_POST['update_cc_num']) || empty($_POST['update_cc_num']))
      $errors[] = __('You must enter your Credit Card number.', 'memberpress');
    elseif(!$this->is_credit_card_valid($_POST['update_cc_num']))
      $errors[] = __('Your credit card number is invalid.', 'memberpress');
    
    if(!isset($_POST['update_cvv_code']) || empty($_POST['update_cvv_code']))
      $errors[] = __('You must enter your CVV code.', 'memberpress');
    
    return $errors;
  }

  /** Actually pushes the account update to the payment processor */
  public function process_update_account_form($sub_id) {
    return $this->process_update_subscription($sub_id);
  }
  
  /** Returns boolean ... whether or not we should be sending in test mode or not */
  public function is_test_mode() {
    return (isset($this->settings->test_mode) and $this->settings->test_mode);
  }

  public function force_ssl() {
    return (isset($this->settings->force_ssl) and ($this->settings->force_ssl == 'on' or $this->settings->force_ssl == true));
  }
  
  protected function send_aim_request($method, $args, $http_method='post') {
    $args = array_merge( array( 'x_login'          => $this->settings->login_name,
                                'x_tran_key'       => $this->settings->transaction_key,
                                'x_version'        => '3.1',
                                'x_delim_data'     => 'TRUE',
                                'x_delim_char'     => '|',
                                'x_relay_response' => 'FALSE', // NOT SURE about this
                                'x_method'         => 'CC' ), $args );

    $remote = array( 'method'      => strtoupper($http_method),
                     'timeout'     => 30,
                     'redirection' => 5,
                     'httpversion' => '1.0',
                     'blocking'    => true,
                     'headers'     => array(),
                     'body'        => $args,
                     'cookies'     => array() );
    
    $this->email_status("Sending AIM request to Authorize.net: \n" . MeprUtils::object_to_string($args, true) . "\n", $this->settings->debug);
    
    $response = wp_remote_post($this->settings->aimUrl, $remote);
    
    if(is_wp_error($response))
      throw new MeprHttpException( sprintf( __( 'You had an HTTP error connecting to %s: %s' , 'memberpress'), $this->name, MeprUtils::object_to_string($response) ) );
    else if($response['response']['code'] != '200')
      throw new MeprHttpException( sprintf( __( 'You had an HTTP error connecting to %s: %s' , 'memberpress'), $this->name, MeprUtils::object_to_string($response) ) );

    $answers = explode('|', $response['body']);
    
    if(empty($answers))
      throw new MeprRemoteException( $response['body'] );
    
    $this->email_status("AIM response from Authorize.net: \n" . MeprUtils::object_to_string($answers, true) . "\n", $this->settings->debug);

    if(intval($answers[0])==1 or intval($answers[0])==4) {
      return array( "response_code" => $answers[0],
                    "response_subcode" => $answers[1],
                    "response_reason_code" => $answers[2],
                    "response_reason_text" => $answers[3],
                    "authorization_code" => $answers[4],
                    "avs_response" => $answers[5],
                    "transaction_id" => $answers[6],
                    "invoice_number" => $answers[7],
                    "description" => $answers[8],
                    "amount" => $answers[9],
                    "method" => $answers[10],
                    "transaction_type" => $answers[11],
                    "customer_id" => $answers[12],
                    "first_name" => $answers[13],
                    "last_name" => $answers[14],
                    "company" => $answers[15],
                    "address" => $answers[16],
                    "city" => $answers[17],
                    "state" => $answers[18],
                    "zip_code" => $answers[19],
                    "country" => $answers[20],
                    "phone" => $answers[21],
                    "fax" => $answers[22],
                    "email_address" => $answers[23],
                    "ship_to_first_name" => $answers[24],
                    "ship_to_last_name" => $answers[25],
                    "ship_to_company" => $answers[26],
                    "ship_to_address" => $answers[27],
                    "ship_to_city" => $answers[28],
                    "ship_to_state" => $answers[29],
                    "ship_to_zip" => $answers[30],
                    "ship_to_country" => $answers[31],
                    "tax" => $answers[32],
                    "duty" => $answers[33],
                    "freight" => $answers[34],
                    "tax_exempt" => $answers[35],
                    "purchase_order_number" => $answers[36],
                    "md5_hash" => $answers[37],
                    "card_code_reason" => $answers[38],
                    "cardholder_authentication_verification_response" => $answers[39],
                    "account_number" => $answers[40],
                    "card_type" => $answers[51],
                    "split_tender_id" => $answers[52],
                    "requested_amount" => $answers[53],
                    "balance_on_card" => $answers[54] );
    }

    throw new MeprRemoteException( $response['body'] );
  }

  protected function send_arb_request($method, $args, $http_method='post') {
    // This method automatically puts the authentication credentials in place
    $args = array_merge( array( "merchantAuthentication" => array(
                                  "name" => $this->settings->login_name,
                                  "transactionKey" => $this->settings->transaction_key
                                )
                              ),
                         $args );
                         
    $content = $this->arb_array_to_xml($method, $args);

    $remote_array = array('method' => strtoupper($http_method),
                          'timeout' => 30,
                          'redirection' => 5,
                          'httpversion' => '1.0',
                          'blocking' => true,
                          'headers' => array('content-type' => 'application/xml'),
                          'body' => $content,
                          'cookies' => array());
    
    $response = wp_remote_post($this->settings->arbUrl, $remote_array);
    
    
    if(is_wp_error($response))
      throw new MeprHttpException( sprintf( __( 'You had an HTTP error connecting to %s: %s' , 'memberpress'), $this->name, MeprUtils::object_to_string($response) ) );
    else if($response['response']['code'] != '200')
      throw new MeprHttpException( sprintf( __( 'You had an HTTP error connecting to %s: %s' , 'memberpress'), $this->name, MeprUtils::object_to_string($response) ) );
    else {
      $answers = $this->simplexml2stdobject(@simplexml_load_string($response['body']));
      
      $this->email_status( "Got this from AuthorizeNet when sending an arb request \n" .
                           MeprUtils::object_to_string($answers, true) .
                           "\nSent with this XML:\n{$content}\n",
                           $this->settings->debug );
    
      if(!empty($answers) and strtolower($answers->messages->resultCode) == 'ok')
        return $answers;
        
      throw new MeprRemoteException( $response['body'] );
    }
  }

  protected function arb_subscription_interval($sub) {
    // Authorize.net doesn't support 'years' or 'weeks' as a unit
    // so we just adjust manually for that case ...
    // and we can't do a longer period with auth.net than
    // one year so just suck it up dude...lol
    if($sub->period_type=='months')
      return array( "length" => $sub->period, "unit" => "months" );
    else if($sub->period_type=='years') {
      $sub->period=1; // Force this down to 1 year
      $sub->store();
      return array( "length" => 12, "unit" => "months" );
    }
    else if($sub->period_type=='weeks')
      return array( "length" => ($sub->period * 7), "unit" => "days" );
  }

  protected function get_order_invoice($sub) {
    return get_post_meta( $sub->ID, self::$order_invoice_str, true );
  }

  protected function create_new_order_invoice($sub) {
    $inv = strtoupper(substr(preg_replace('/\./','',uniqid('',true)),-20));
    update_post_meta($sub->ID, self::$order_invoice_str, $inv);
    return $inv;
  }

  // The simplexml objects are not cool ...
  // we want something more vanilla
  protected function simplexml2stdobject($obj) {
    $array = array();
    foreach( (array)$obj as $k => $v )
      $array[$k] = ($v instanceof SimpleXMLElement) ? $this->simplexml2stdobject($v) : $v;
    return (object)$array;
  }

  protected function arb_array_to_xml($method, $array, $level=0) {
    if($level==0) {
      $xml = "<?xml version=\"1.0\" encoding=\"utf-8\" ?>\n";
      $xml .= "<{$method} xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">\n";
    }
    else
      $xml = '';

    foreach($array as $key => $value ) {
      // Print indentions
      for($i=0; $i < $level+1; $i++) { $xml .= "  "; }

      // Print open tag (looks like we don't need 
      // to worry about attributes with this schema)
      $xml .= "<{$key}>";

      // Print value or recursively render sub arrays
      if(is_array($value)) {
        $xml .= "\n";
        $xml .= $this->arb_array_to_xml($method,$value,$level+1);
        // Print indentions for end tag
        for($i=0; $i < $level+1; $i++) { $xml .= "  "; }
      }
      else
        $xml .= $value;

      // Print End tag
      $xml .= "</{$key}>\n";
    }

    if($level==0)
      $xml .= "</{$method}>\n";

    return $xml;
  }
}
